<?php

/**
 * Script provide ability for mass password reset of Plesk entities.
 *
 * Requirements: script supports PHP4/PHP5 in case where installed PHP4 only
 */
define('APP_PATH', dirname(__FILE__));
define('DEBUG', 0); // allow to dump sql logs to output
define('PRE_UPGRADE_SCRIPT_VERSION', '12.0.18.0'); //script version
define('PLESK_VERSION', '12.0.18'); // latest Plesk version
define('AI_VERSION', '3.16.22'); // latest autoinstaller version

if (!defined('PHP_EOL')) {
    define('PHP_EOL', "\n", true);
}

$phpType = php_sapi_name();
if (substr($phpType, 0, 3) == 'cgi') {
    //:INFO: set max execution time 1hr
    @set_time_limit(3600);
}

//:INFO: Validate options
$options = new GetOpt();
$options->validate();

if (PleskPasswordChanger::isCbmIntegration() && PleskPasswordChanger::isCbmLocal()) {
	if (PleskVersion::is10_4()) {
		//includeBilling10Files();
		define("CLI_INTERFACE", true);
		
		// MB specific initailization
		if ("WIN" == strtoupper(substr(PHP_OS, 0, 3))) {
			define("MB_LIBPATH",  Util::getPleskRootPath() . "billing\lib");
		} else {
			define("MB_LIBPATH", '/opt/plesk-billing/lib');
		}
		
		define("APP_LIB", "lib-billing");
		$mbpath = array(MB_LIBPATH);
		require_once(MB_LIBPATH . '/lib-pkg/parsefunction.php');
		define ("ACCESS_TYPE", NON_USER_APP);
		require_once(MB_LIBPATH . '/lib-tk/include/common.php');
		
		require_once(MB_LIBPATH . '/lib-tk/include/crypt/crypt.php');
		
		define("SUCCESS_CODE", 0);
		define("FAILURE_CODE", 254);
		
		require_once 'Loader.php';
		Loader::registerAutoload();
		
		require_once("object_managers/ClientContactManager.php");
		require_once("object_managers/ResellerManager.php");
		require_once("object_managers/AdminManager.php");
	} elseif (PleskVersion::is11_0()) {
		if ("WIN" == strtoupper(substr(PHP_OS, 0, 3))) {
			require_once(Util::getPleskRootPath() . 'billing\scripts\cli_prepender.php.inc');
			require_once(Util::getPleskRootPath() . 'billing\lib\billing-libs\object_managers\CryptManager.php');
			require_once(Util::getPleskRootPath() . 'billing\lib\billing-libs\Billing\Manager\ContactManager.php');
			require_once(Util::getPleskRootPath() . 'billing\lib\billing-libs\object_managers\ResellerManager.php');
		} else {
			require_once('/usr/share/plesk-billing/cli_prepender.php.inc');
			require_once('/opt/plesk-billing/lib/billing-libs/object_managers/CryptManager.php');
			require_once('/opt/plesk-billing/lib/billing-libs/Billing/Manager/ContactManager.php');
			require_once('/opt/plesk-billing/lib/billing-libs/object_managers/ResellerManager.php');
		}
	} elseif (PleskVersion::is11_5_or_above()) {
		if ("WIN" == strtoupper(substr(PHP_OS, 0, 3))) {
			require_once(Util::getPleskRootPath() . 'billing\scripts\cli_prepender.php.inc');
			require_once(Util::getPleskRootPath() . 'billing\lib\billing-libs\Billing\Manager\CryptManager.php');
			require_once(Util::getPleskRootPath() . 'billing\lib\billing-libs\Billing\Manager\ContactManager.php');
			require_once(Util::getPleskRootPath() . 'billing\lib\billing-libs\Billing\Manager\ResellerManager.php');
		} else {
			require_once('/usr/share/plesk-billing/cli_prepender.php.inc');
			require_once('/opt/plesk-billing/lib/billing-libs/Billing/Manager/CryptManager.php');
			require_once('/opt/plesk-billing/lib/billing-libs/Billing/Manager/ContactManager.php');
			require_once('/opt/plesk-billing/lib/billing-libs/Billing/Manager/ResellerManager.php');
		}

	}
}

class PleskPasswordChanger
{
	private $plesk_dir = '';
	private $plesk_password_strength = '';

	public function __construct()
	{
		$this->getPleskDir();
	}
	
	public function getPleskDir()
	{
		if (Util::isLinux()) {
			$this->plesk_dir = Util::getPleskRootPath();
		} else {
			if (PleskVersion::is8x()) {
				$this->plesk_dir = Util::regPleskQuery('PRODUCT_ROOT_D', true) . "admin\\";
			} else {
				$this->plesk_dir = Util::regPleskQuery('PRODUCT_ROOT_D', true);
			}
		}
	}

	public function getPleskPasswordStrength()
	{
		if (PleskVersion::is10x_or_above()) // generate the strongest passwords if we have for Plesk 10+
		{
			$this->plesk_password_strength = 'Stronger';
			return 0;
		}

		$this->plesk_password_strength = 'Mediocre'; // generate medium passwords otherwise
		return 0;
	}
	
	public function changeAllPasswords($new_admin_password = '')
	{
		global $options;
		
		$this->logToCsv('owner_name', 'owner_type', 'owner_login', 'owner_email', 'domain', 'entity_type', 'entity_login', 'entity_id', 'new_password');
		
		if (in_array('--cbm', $options->_argv)) {
			$this->changeForCbmUsers();
		}
		if (!in_array('--exclude-resellers', $options->_argv) && in_array('--resellers', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->changeForResellers();
		}
		if (!in_array('--exclude-clients', $options->_argv) && in_array('--clients', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->changeForClients();
		}
		if (in_array('--domainadmins', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->changeForDomainAdmins();
		}
		if (in_array('--users', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->changeForUsers();
		}
		if (in_array('--domains', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->changeForDomains();
		}
		if (in_array('--subdomains', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->changeForSubDomains();
		}
		if (in_array('--dbusers', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->changeForDatabaseUsersAccounts();
		}
		if (in_array('--additionalftpaccounts', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->changeForAdditionalFTPaccounts();
		}
		if (in_array('--mailaccounts', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->changeForMailAccounts();
		}
		if (in_array('--webusers', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->changeForWebUsers();
		}
		if (in_array('--pdusers', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->changeForProtectedDirectoriesUsers();
		}
		if (in_array('--clean-up-sessions', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->cleanUpSessions();
		}
		if (in_array('--additionaladmins', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->changeForAdditionalAdmins();
		}
		if (in_array('--apsc', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->changeForApsc();
		}
		if (in_array('--admin', $options->_argv) || in_array('--all', $options->_argv)) {
			$this->changeForAdmin($new_admin_password);
		}
	}
	
	public function changeForApsc()
	{
		if (Util::isLinux() && PleskVersion::is11_5_or_above()) {
			Log::step("Change password for apsc database...", true);
			
			$cmd = "{$this->plesk_dir}/bin/sw-engine-pleskrun {$this->plesk_dir}/admin/plib/scripts/check_apsc_connection.php";
			$output = Util::exec($cmd, $code);
		
			if ($output === 'connection ok') {
				Log::info('apsc connection is OK. Skip password changing.');
			} else {
				$newPassword = $this->getNewPassword();
				
				$db = PleskDb::getInstance();
				$sql = "SET PASSWORD FOR 'apsc'@'localhost' = PASSWORD('{$newPassword}')";
				$result = $db->query($sql);
				$db->close();
				
				$cmd = "{$this->plesk_dir}/bin/sw-engine-pleskrun {$this->plesk_dir}/admin/plib/scripts/register_apsc_database.php --register -host 'localhost' -port 3306 -database 'apsc' -login 'apsc' -password '{$newPassword}'";
				$output = Util::exec($cmd, $code);
				if ($output === 'APSC database has been registered successfully') {
					Log::info('apsc database login: apsc New password: ' . $newPassword);
					return;
				}
				Log::warning("Update password for apsc database is failed! Command failed: {$cmd}: {$output}");
			}
		}
	}
	
	public function cleanUpSessions()
	{
		Log::step("Clean up all user sessions in Plesk database...", true);
		
		$db = PleskDb::getInstance();
		$sql = "delete from sessions";
		$resellers = $db->query($sql);
		$db->close();
				
	}
	
	static function isCbmIntegration()
	{
		$db = PleskDb::getInstance();
		$sql = "select 1 from cl_param where param='integration-api-version'";
		$integration = $db->fetchAll($sql);
		$db->close();
	
		if ($integration) {
			return True;
		}
	
		return False;
	}
	
	static function isCbmLocal()
	{
		if (Util::isWindows()) {
			$billing = is_dir(Util::getPleskRootPath() . "billing\lib");
		} else {
			$billing = is_dir('/opt/plesk-billing/lib');
		}
		
		if ($billing) {
			return True;
		}
		
		return False;
	}
	
	static function getCbmHash($password)
	{
		if (PleskVersion::is10_4()) {
			return getPasswordHash($password);
		} elseif (PleskVersion::is11x()) {
			return CryptManager::getPasswordHash($password);
		} else {
			Log::warning('Unsupported version detected! Only 10.4 and 11.x are supported. Passwords for CBM users will be not changed.');
			return;
		}
	}
	
	public function changeForCbmUsers()
	{
		if (!self::isCbmIntegration()) {
			Log::warning('There is no integration with Customer & Business Manager. Passwords for CBM users will be not changed.');
			return;
		}
		
		if (!self::isCbmLocal()) {
			Log::warning('Customer & Business Manager is not local! Passwords for CBM users will be not changed.');
			return;
		}
		
		Log::step("Check Customer & Business Manager version...", true);
		
		if (PleskVersion::is10_4()) { 
			$contactList = Factory()->ClientContactList();
		} elseif (PleskVersion::is11x()) {
			$contactList = Factory()->ContactList();
		} else {
			Log::warning('Unsupported version detected! Only 10.4 and 11.x are supported. Passwords for CBM users will be not changed.');
			return;
		}
		
		Log::step("Change passwords for Customer & Business Manager users...", true);
		
		foreach ($contactList as $contact) {
			$newPassword = $this->getNewPassword();
			$contact->plainPassword = $newPassword;
			$contact->contact_password = self::getCbmHash($newPassword);
            ClientContactManager::update($contact, $newPassword);
            if (PleskVersion::is11x()) {
                eval('use Billing\Manager\ContactManager; ContactManager::update($contact);');
            }
            	
            $client = Factory()->Client($contact->client_id);
            $entityType = 'CbmClient';
            if ($client->isReseller()) {
            	$entityType = 'CbmReseller';
            	$reseller = $client->getReseller();
            	//$reseller = Factory()->Reseller($contact->client_id);
            	//var_dump($reseller);
           		
            	if (PleskVersion::is11x()) {
           			$reseller->plainPassword = $newPassword;
            	}
           		$reseller->password = self::getCbmHash($newPassword);
           		ResellerManager::update($reseller, $newPassword);
           		
           	}
           	
			$this->logToCsv('', '', '', '', '', $entityType, $contact->contact_username, $contact->id, $newPassword);
			Log::info('CBM user login: ' . $contact->contact_username . ' Email: ' . $contact->contact_email . ' New password: ' . $newPassword);
		}
	}
	
	function changeForCbmAdmin($new_admin_password)
	{
		if (PleskPasswordChanger::isCbmIntegration() && PleskPasswordChanger::isCbmLocal()) {
			$cbmAdmin = Factory()->Admin('admin_ezorder', 'admin_username');
			if (PleskVersion::is11x()) {
				$cbmAdmin->plainPassword = $newPassword;
			}
			$cbmAdmin->admin_password = self::getCbmHash($new_admin_password);
			AdminManager::update($cbmAdmin, false, $new_admin_password);
				
			$cbmAdmin = Factory()->Admin('admin', 'admin_username');
			if (PleskVersion::is11x()) {
				$cbmAdmin->plainPassword = $newPassword;
			}
			$cbmAdmin->admin_password = self::getCbmHash($new_admin_password);
			AdminManager::update($cbmAdmin, false, $new_admin_password);
		}
	}
	
	function changeForAdmin($new_admin_password)
	{
		global $options;
		
		if ($new_admin_password == '') {
			$new_admin_password = $this->getNewPassword();
		}
		Log::step("Change password for admin...", true);
		$this->getPleskDir();
		
		if (Util::isLinux()) {
			putenv("PSA_PASSWORD=$new_admin_password");
			$cmd = $this->plesk_dir . "/bin/init_conf -u -passwd ''";
			Util::exec($cmd, $code);
		} else {
			$cmd = '"' . Util::regPleskQuery('PRODUCT_ROOT_D', true) . 'admin\bin\plesksrvclient.exe" -set ' . $this->wrapNewPassword($new_admin_password) . ' true';
			Util::exec($cmd, $code);
		}
		if ($code === 0) {
			$this->logToCsv('', '', '', '', '', 'admin', 'admin', '', $new_admin_password);
			Log::info('Admin new password: ' . $new_admin_password);
			$options->_adminDbPasswd = $new_admin_password;
		} else {
			Log::warning('Unable to execute: ' . $cmd . ' Return code is: ' . $code);
		}
	}
	
	public function changeForAdditionalAdmins()
	{
		if (Util::isLinux() && !PleskVersion::is11_5_or_above()) {
			return;
		}
			
		Log::step("Change password for additional administrators accounts...", true);
		$this->getPleskDir();
	
		$db = PleskDb::getInstance();
		$sql = "SELECT id, login, aemail FROM admin_aliases";
		$addadmins = $db->fetchAll($sql);
		$db->close();
	
		if (Util::isLinux()) {
			$cmd = $this->plesk_dir . '/bin/admin_alias'; // its not impleented yet
		} else {
			$cmd = '"' . $this->plesk_dir . 'bin\admin_alias.exe"';
		}
		
		if  (PleskVersion::is8x() || PleskVersion::is9x()) {
			$db = PleskDb::getInstance();
			foreach ($addadmins as $addadmin) {
				$newPassword = $this->getNewPassword();
				
				$update_query = "UPDATE admin_aliases SET passwd ='" . $newPassword . "' WHERE id = " . $addadmin['id'] ;
				$result = $db->query($update_query);
				if (empty($result)) {
					$this->logToCsv('', '', '', '', '', 'additionaladmin', $addadmin['login'], $addadmin['id'], $newPassword);
					Log::info('Additional administrator login: ' . $addadmin['login'] . ' Email: ' . $addadmin['aemail'] . ' New password: ' . $newPassword);
				} else {
					Log::warning('Unable to execute SQL query: ' . $update_query);
					var_dump($result);
				}
				
			}
			$db->close();
			
		} else {
			foreach ($addadmins as $addadmin) {
				$newPassword = $this->getNewPassword();
				Util::exec($cmd . ' -u ' . $addadmin['login'] . ' -passwd ' . $this->wrapNewPassword($newPassword), $code);
				if ($code === 0) {
					$this->logToCsv('', '', '', '', '', 'additionaladmin', $addadmin['login'], $addadmin['id'], $newPassword);
					Log::info('Additional administrator login: ' . $addadmin['login'] . ' Email: ' . $addadmin['aemail'] . ' New password: ' . $newPassword);
				} else {
					Log::warning('Unable to execute: ' . $cmd . ' -u ' . $addadmin['login'] . ' -passwd ' . $newPassword . ' Return code is: ' . $code);
				}
			}
		}
	
	}
	
	public function changeForResellers()
	{
		if (PleskVersion::is8x()) {
			return;
		}
			
		Log::step("Change password for resellers...", true);
		$this->getPleskDir();
		
		$db = PleskDb::getInstance();
		$sql = "SELECT clients.id, clients.login, clients.email FROM clients WHERE type='reseller'";
		$resellers = $db->fetchAll($sql);
		$db->close();
		
		if (Util::isLinux()) {
			$cmd = $this->plesk_dir . '/bin/reseller';
		} else {
			$cmd = '"' . $this->plesk_dir . 'bin\reseller.exe"';
		}
		
		foreach ($resellers as $reseller) {
			$newPassword = $this->getNewPassword();
			Util::exec($cmd . ' -u ' . $reseller['login'] . ' -passwd ' . $this->wrapNewPassword($newPassword), $code);
			if ($code === 0) {
				$this->logToCsv('', '', '', '', '', 'reseller', $reseller['login'], $reseller['id'], $newPassword);
				Log::info('Reseller login: ' . $reseller['login'] . ' Email: ' . $reseller['email'] . ' New password: ' . $newPassword);
			} else {
				Log::warning('Unable to execute: ' . $cmd . ' -u ' . $reseller['login'] . ' -passwd ' . $newPassword . ' Return code is: ' . $code);
			}
		}
		
	}
	
	public function changeForClients()
	{
		Log::step("Change password for clients...", true);
		$this->getPleskDir();
		
		$db = PleskDb::getInstance();
		if (PleskVersion::is8x()) {
			$sql = "SELECT clients.id, clients.login, clients.email FROM clients";
		} else { 
			$sql = "SELECT cl.id, res.login AS owner_login, res.cname AS owner_name, res.email AS owner_email, res.type AS owner_type, cl.id, cl.login, cl.email FROM clients as cl, clients as res WHERE cl.parent_id = res.id and cl.type = 'client'";
		}
		$clients = $db->fetchAll($sql);
		$db->close();
		
		if (PleskVersion::is10x_or_above()) {
			$entity_type = 'customer';
		} else {
			$entity_type = 'client';
		}
		
		if (Util::isLinux()) {
			$cmd = $this->plesk_dir . '/bin/client';
		} else {
			$cmd = '"' . $this->plesk_dir . 'bin\client.exe"';
		}
		
		foreach ($clients as $client) {
			$newPassword = $this->getNewPassword();
			Util::exec($cmd . ' -u ' . $client['login'] . '  -passwd ' . $this->wrapNewPassword($newPassword), $code);
			if ($code === 0) {
				if (isset($client['owner_login'])) {
					$this->logToCsv($client['owner_name'], $client['owner_type'], $client['owner_login'], $client['owner_email'], '', $entity_type, $client['login'], $client['id'], $newPassword);
				} else {
					$this->logToCsv('', '', '', '', '', $entity_type, $client['login'], $client['id'], $newPassword);
				}
				Log::info('Client login: ' . $client['login'] . ' Email: ' . $client['email'] . ' New password: ' . $newPassword);
			} else {
				Log::warning('Unable to execute: ' . $cmd . ' -u ' . $client['login'] . ' -passwd ' . $newPassword . ' Return code is: ' . $code);
			}
		}
	}
	
	public function changeForDomainAdmins()
	{
		if (PleskVersion::is10x_or_above()) {
			return;
		}
	
		Log::step("Change password for domain administrators...", true);
		$this->getPleskDir();
		
		$db = PleskDb::getInstance();
		if (PleskVersion::is8x()) {
			$sql = "SELECT domains.id, name, clients.login AS owner_login, clients.cname AS owner_name, 'client' AS owner_type, clients.email AS owner_email FROM dom_level_usrs, domains, clients WHERE dom_level_usrs.dom_id = domains.id AND clients.id = domains.cl_id";
		} else {
			$sql = "SELECT domains.id, name, clients.login AS owner_login, clients.cname AS owner_name, clients.type AS owner_type, clients.email AS owner_email FROM dom_level_usrs, domains, clients WHERE dom_level_usrs.dom_id = domains.id AND clients.id = domains.cl_id";
		} 
		$domains = $db->fetchAll($sql);
		$db->close();
		
		if (Util::isLinux()) {
			$cmd = $this->plesk_dir . '/bin/domadmin';
		} else {
			$cmd = '"' . $this->plesk_dir . 'bin\domadmin.exe"';
		}
		
		foreach ($domains as $domain) {
			$newPassword = $this->getNewPassword();
			Util::exec($cmd . ' -u ' . $domain['name'] . ' -passwd ' . $this->wrapNewPassword($newPassword), $code);
			if ($code === 0) {
				$this->logToCsv($domain['owner_name'], $domain['owner_type'], $domain['owner_login'], $domain['owner_email'], $domain['name'], 'domain administrator', $domain['name'], $domain['id'], $newPassword);
				Log::info('Domain administrator: ' . $domain['name'] . ' New password: ' . $newPassword);
			} else {
				Log::warning('Unable to execute: ' . $cmd . ' -u ' . $domain['name'] . ' -passwd ' . $newPassword . ' Return code is: ' . $code);
			}
		}
	}
	
	public function changeForUsers()
	{
		if (PleskVersion::is9x() || PleskVersion::is8x()) {
			return;
		}
		
		// select  login, email from smb_users where smb_users.login not in (select login from clients)
		Log::step("Change password for users...", true);
		$this->getPleskDir();
		
		$db = PleskDb::getInstance();
		$sql = "SELECT smb_users.id, smb_users.login, smb_users.email, clients.login AS owner_login, clients.cname AS owner_name, clients.type AS owner_type, clients.email AS owner_email FROM smb_users, clients WHERE smb_users.ownerId = clients.id  AND smb_users.login NOT IN (SELECT login FROM clients)";
		$users = $db->fetchAll($sql);
		$db->close();
		
		if (Util::isLinux()) {
			$cmd = $this->plesk_dir . '/bin/user';
		} else {
			$cmd = '"' . $this->plesk_dir . 'bin\user.exe"';
		}
		
		foreach ($users as $user) {
			$newPassword = $this->getNewPassword();
			Util::exec($cmd . ' -u ' . $user['login'] . ' -passwd ' . $this->wrapNewPassword($newPassword), $code);
			if ($code === 0) {
				$this->logToCsv($user['owner_name'], $user['owner_type'], $user['owner_login'], $user['owner_email'], '', 'hosting panel user', $user['login'], $user['id'], $newPassword);
				Log::info('Hosting Panel User: ' . $user['login'] . ' New password: ' . $newPassword);
			} else {
				Log::warning('Unable to execute: ' . $cmd . ' -u ' . $user['login'] . ' -passwd ' . $newPassword . ' Return code is: ' . $code);
			}
		}
	}
	
	public function changeForDomains()
	{
		Log::step("Change password for FTP users of domains...", true);
		$this->getPleskDir();
		
		$db = PleskDb::getInstance();
		if (PleskVersion::is8x()) {
			$sql = "SELECT domains.id, domains.name, sys_users.login, clients.login AS owner_login, clients.cname AS owner_name, 'client' AS owner_type, clients.email AS owner_email FROM domains, hosting, sys_users, clients WHERE domains.id = hosting.dom_id and hosting.sys_user_id = sys_users.id and clients.id = domains.cl_id";
		} elseif (PleskVersion::is9x()) { 
			$sql = "SELECT domains.id, domains.name, sys_users.login, clients.login AS owner_login, clients.cname AS owner_name, clients.type AS owner_type, clients.email AS owner_email FROM domains, hosting, sys_users, clients WHERE domains.id = hosting.dom_id and hosting.sys_user_id = sys_users.id and clients.id = domains.cl_id";
		} elseif (PleskVersion::is10x_or_above()) { 
			$sql = "SELECT domains.id, domains.name, sys_users.login, clients.login AS owner_login, clients.cname AS owner_name, clients.type AS owner_type, clients.email AS owner_email FROM domains, hosting, sys_users, clients WHERE domains.id = hosting.dom_id and hosting.sys_user_id = sys_users.id and clients.id = domains.cl_id group by sys_users.login";
			if (Util::isWindows()) {
				$dbprovider = Util::regPleskQuery('PLESK_DATABASE_PROVIDER_NAME');
				if ($dbprovider == 'MSSQL') {
					$sql = "SELECT domains.id, domains.name, sys_users.login, clients.login AS owner_login, clients.cname AS owner_name, clients.type AS owner_type, clients.email AS owner_email FROM domains, hosting, sys_users, clients WHERE domains.id = hosting.dom_id and hosting.sys_user_id = sys_users.id and clients.id = domains.cl_id group by sys_users.login, domains.id, domains.name, sys_users.login, clients.login, clients.cname, clients.type, clients.email";
				}
			}
		} 
		$domains = $db->fetchAll($sql);
		$db->close();
		
		if (Util::isLinux()) {
			$cmd = $this->plesk_dir . '/bin/domain';
		} else {
			$cmd = '"' . $this->plesk_dir . 'bin\domain.exe"';
		}
		
		$duplicates = array();
		
		foreach ($domains as $domain) {
			if ( in_array($domain['login'], $duplicates, false)) { # need for MSSQL, there is no ability to group by
				continue;
			} else {
				$duplicates[] = $domain['login'];
			}
			
			$newPassword = $this->getNewPassword();
			Util::exec($cmd . ' -u ' . $domain['name'] . ' -passwd ' . $this->wrapNewPassword($newPassword), $code);
			if ($code === 0) {
				$this->logToCsv($domain['owner_name'], $domain['owner_type'], $domain['owner_login'], $domain['owner_email'], $domain['name'], 'domain FTP account', $domain['login'], $domain['id'], $newPassword);
				Log::info('FTP user ' . $domain['login'] . ' for domain ' . $domain['name'] . ' New password: ' . $newPassword);
			} else {
				Log::warning('Unable to execute: ' . $cmd . ' -u ' . $domain['name'] . ' -passwd ' . $newPassword . ' Return code is: ' . $code);
			}
		}
	}
	
	public function changeForAdditionalFTPaccounts()
	{
		if (Util::isLinux() && (PleskVersion::is8x() || PleskVersion::is9x())) {
			return;
		}
		
		Log::step("Change password for additional FTP accounts...", true);
		$this->getPleskDir();
		
		$db = PleskDb::getInstance();
		if (PleskVersion::is8x()) {
			$sql = "SELECT sys_users.id, sys_users.login, domains.name, clients.login AS owner_login, clients.cname AS owner_name, 'client' AS owner_type, clients.email AS owner_email FROM ftp_users, sys_users, domains, clients where ftp_users.sys_user_id = sys_users.id and ftp_users.dom_id = domains.id and clients.id = domains.cl_id";
		} else {
			$sql = "SELECT sys_users.id, sys_users.login, domains.name, clients.login AS owner_login, clients.cname AS owner_name, clients.type AS owner_type, clients.email AS owner_email FROM ftp_users, sys_users, domains, clients where ftp_users.sys_user_id = sys_users.id and ftp_users.dom_id = domains.id and clients.id = domains.cl_id";
		}
		$ftpaccounts = $db->fetchAll($sql);
		$db->close();
		
		if (Util::isLinux()) {
			$cmd = $this->plesk_dir . '/bin/ftpsubaccount';
		} else {
			$cmd = '"' . $this->plesk_dir . 'bin\ftpsubaccount.exe"';
		}
		
		foreach ($ftpaccounts as $account) {
			$newPassword = $this->getNewPassword();
			Util::exec($cmd . ' -u ' . $account['login'] . ' -passwd ' . $this->wrapNewPassword($newPassword) . ' -domain ' . $account['name'], $code);
			if ($code === 0) {
				$this->logToCsv($account['owner_name'], $account['owner_type'], $account['owner_login'], $account['owner_email'], $account['name'], 'additional FTP account', $account['login'], $account['id'], $newPassword);
				Log::info('Domain: ' . $account['name'] . ' Additional FTP account: ' . $account['login'] . ' New password: ' . $newPassword);
			} else {
				Log::warning('Unable to execute: ' . $cmd . ' -u ' . $account['login'] . ' -passwd ' . $newPassword . ' -domain ' . $account['name'] . ' Return code is: ' . $code);
			}
		}
		
	}
	
	public function changeForSubDomains()
	{
		if (PleskVersion::is10x_or_above()) {
			return;
		}
		
		Log::step("Change password for FTP users of subdomains...", true);
		$this->getPleskDir();
		
		$db = PleskDb::getInstance();
		if (PleskVersion::is8x()) {
			$sql = "SELECT subdomains.id, subdomains.name AS sub, domains.name AS dom, sys_users.login, clients.login AS owner_login, clients.cname AS owner_name, 'client' AS owner_type, clients.email AS owner_email FROM subdomains, domains, sys_users, clients WHERE subdomains.dom_id=domains.id AND sys_user_type='native' AND sys_users.id = subdomains.sys_user_id and clients.id = domains.cl_id";
		} else {
			$sql = "SELECT subdomains.id, subdomains.name AS sub, domains.name AS dom, sys_users.login, clients.login AS owner_login, clients.cname AS owner_name, clients.type AS owner_type, clients.email AS owner_email FROM subdomains, domains, sys_users, clients WHERE subdomains.dom_id=domains.id AND sys_user_type='native' AND sys_users.id = subdomains.sys_user_id and clients.id = domains.cl_id";
		}
		$subdomains = $db->fetchAll($sql);
		$db->close();
		
		if (Util::isLinux()) {
			$cmd = $this->plesk_dir . '/bin/subdomain';
		} else {
			$cmd = '"' . $this->plesk_dir . 'bin\subdomain.exe"';
		}
		
		foreach ($subdomains as $subdomain) {
			$newPassword = $this->getNewPassword();
			Util::exec($cmd . ' -u ' . $subdomain['sub'] . ' -d ' . $subdomain['dom'] . ' -passwd ' . $this->wrapNewPassword($newPassword), $code);
			if ($code === 0) {
				$this->logToCsv($subdomain['owner_name'], $subdomain['owner_type'], $subdomain['owner_login'], $subdomain['owner_email'], $subdomain['dom'], 'subdomain', $subdomain['login'], $subdomain['id'], $newPassword);
				Log::info('FTP user ' . $subdomain['login']  . ' for subdomain ' . $subdomain['sub'] . '.' . $subdomain['dom'] . ' New password: ' . $newPassword);
			} else {
				Log::warning('Unable to execute: ' . $cmd . ' -u ' . $subdomain['sub'] . ' -d ' . $subdomain['dom'] . ' -passwd ' . $newPassword . ' Return code is: ' . $code);
			}
		}
	}
	
	function changeForWebUsers()
	{
		Log::step("Change password for web users of domains...", true);
		$this->getPleskDir();
		
		$db = PleskDb::getInstance();
		if (PleskVersion::is8x()) {
			$sql = "SELECT sys_users.id, sys_users.login, domains.name, clients.login AS owner_login, clients.cname AS owner_name, 'client' AS owner_type, clients.email AS owner_email FROM web_users, sys_users, clients, domains WHERE web_users.sys_user_id = sys_users.id AND web_users.dom_id = domains.id and clients.id = domains.cl_id";
		} else {
			$sql = "SELECT sys_users.id, sys_users.login, domains.name, clients.login AS owner_login, clients.cname AS owner_name, clients.type AS owner_type, clients.email AS owner_email FROM web_users, sys_users, clients, domains WHERE web_users.sys_user_id = sys_users.id AND web_users.dom_id = domains.id and clients.id = domains.cl_id";
		}	
		$webusers = $db->fetchAll($sql);
		$db->close();
		
		if (Util::isLinux()) {
			$cmd = $this->plesk_dir . '/bin/webuser';
		} else {
			$cmd = '"' . $this->plesk_dir . 'bin\webuser.exe"';
		}
		
		foreach ($webusers as $webuser) {
			$newPassword = $this->getNewPassword();
			Util::exec($cmd . ' -u ' . $webuser['login'] . ' -domain ' . $webuser['name'] . ' -passwd ' . $this->wrapNewPassword($newPassword), $code);
			if ($code === 0) {
				$this->logToCsv($webuser['owner_name'], $webuser['owner_type'], $webuser['owner_login'], $webuser['owner_email'], $webuser['name'], 'web user', $webuser['login'], $webuser['id'], $newPassword);
				Log::info('Web user ' . $webuser['login']  . ' for domain ' . $webuser['name'] . ' New password: ' . $newPassword);
			} else {
				Log::warning('Unable to execute: ' . $cmd . ' -u ' . $webuser['login'] . ' -domain ' . $webuser['name'] . ' -passwd ' . $newPassword . ' Return code is: ' . $code);
			}
		}
		
	}
	
	public function changeForMailAccounts()
	{
		Log::step("Change password for mail accounts...", true);
		$this->getPleskDir();
		
		$db = PleskDb::getInstance();
		if (PleskVersion::is8x()) {
			$sql = "SELECT mail.id, mail.mail_name, domains.name, clients.login AS owner_login, clients.cname AS owner_name, 'client' AS owner_type, clients.email AS owner_email FROM mail, domains, clients WHERE mail.dom_id=domains.id and clients.id = domains.cl_id";
		} elseif (PleskVersion::is9x()) {
			$sql = "SELECT mail.id, mail.mail_name, domains.name, clients.login AS owner_login, clients.cname AS owner_name, clients.type AS owner_type, clients.email AS owner_email FROM mail, domains, clients WHERE mail.dom_id=domains.id and clients.id = domains.cl_id";
		} else {
			$sql = "SELECT mail.id, mail.mail_name, domains.name, clients.login AS owner_login, clients.cname AS owner_name, clients.type AS owner_type, clients.email AS owner_email FROM mail, domains, clients WHERE mail.dom_id=domains.id and (mail.userId = 0 OR mail.userId is NULL) and clients.id = domains.cl_id";
		}
		$mails = $db->fetchAll($sql);
		$db->close();
		
		if (Util::isLinux()) {
			$cmd = $this->plesk_dir . '/bin/mail';
		} else {
			$cmd = '"' . $this->plesk_dir . 'bin\mail.exe"';
		}
		
		foreach ($mails as $account) {
			$newPassword = $this->getNewPassword();
			Util::exec($cmd . ' -u ' . $account['mail_name'] . '@' . $account['name'] . ' -passwd ' . $this->wrapNewPassword($newPassword), $code);
			if ($code === 0) {
				$this->logToCsv($account['owner_name'], $account['owner_type'], $account['owner_login'], $account['owner_email'], $account['name'], 'mail account', $account['mail_name'], $account['id'], $newPassword);
				Log::info('Mail account: ' . $account['mail_name'] . '@' . $account['name'] . ' New password: ' . $newPassword);
			} else {
				Log::warning('Unable to execute: ' . $cmd . ' -u ' . $account['mail_name'] . '@' . $account['name'] . ' -passwd ' . $newPassword . ' Return code is: ' . $code);
			}
		}
	}
	
public function changeForProtectedDirectoriesUsers()
	{
		if (PleskVersion::is11x_or_above()) {
			return;
		}
		
		Log::step("Change password for users of protected directories of the domains...", true);
		$this->getPleskDir();
				
		$db = PleskDb::getInstance();
		if (PleskVersion::is8x()) {
			$sql = "SELECT pd_users.id, pd_users.login, domains.name, clients.login AS owner_login, clients.pname AS owner_name, 'client' AS owner_type, clients.email as owner_email, protected_dirs.path  FROM pd_users, domains, clients, protected_dirs WHERE pd_users.pd_id = protected_dirs.id AND protected_dirs.dom_id = domains.id AND clients.id = domains.cl_id";
		} else {
			$sql = "SELECT pd_users.id, pd_users.login, domains.name, clients.login AS owner_login, clients.pname AS owner_name, clients.type AS owner_type, clients.email as owner_email, protected_dirs.path  FROM pd_users, domains, clients, protected_dirs WHERE pd_users.pd_id = protected_dirs.id AND protected_dirs.dom_id = domains.id AND clients.id = domains.cl_id";
		}	
		$protdirs = $db->fetchAll($sql);
		$db->close();
		
		if (Util::isLinux()) {
			$cmd = $this->plesk_dir . '/bin/protdir';
		} else {
			$cmd = '"' . $this->plesk_dir . 'bin\protdir.exe"';
		}
		
		foreach ($protdirs as $protdir) {
			$newPassword = $this->getNewPassword();
			Util::exec($cmd . ' -u ' . $protdir['path'] . ' -domain ' . $protdir['name'] . ' -update_user ' . $protdir['login'] . ' -passwd ' . $this->wrapNewPassword($newPassword), $code);
			if ($code === 0) {
				$this->logToCsv($protdir['owner_name'], $protdir['owner_type'], $protdir['owner_login'], $protdir['owner_email'], $protdir['name'], 'pduser', $protdir['login'], $protdir['id'], $newPassword);
				Log::info('PD user ' . $protdir['login']  . ' for domain ' . $protdir['name'] . ' New password: ' . $newPassword);
			} else {
				Log::warning('Unable to execute: ' . $cmd . ' -u ' . $protdir['path'] . ' -domain ' . $protdir['name'] . ' -update_user ' . $protdir['login'] . ' -passwd ' . $newPassword . ' Return code is: ' . $code);
			}
		}
		
	}
	
	public function changeForDatabaseUsersAccounts()
	{
		Log::step("Change password for database users...", true);
		$this->getPleskDir();
		
		$db = PleskDb::getInstance();
		
		if (PleskVersion::is8x()) {
			$sql = "SELECT domains.name AS domain_name, clients.login AS owner_login, clients.pname AS owner_name, 'client' AS owner_type, clients.email AS owner_email, db_users.id, data_bases.name, data_bases.type, DatabaseServers.host, DatabaseServers.port, db_users.login FROM domains, clients, db_users, data_bases, DatabaseServers WHERE db_users.db_id = data_bases.id AND data_bases.dom_id = domains.id AND clients.id = domains.cl_id AND DatabaseServers.id = data_bases.db_server_id";
		} else {
			$sql = "SELECT domains.name AS domain_name, clients.login AS owner_login, clients.pname AS owner_name, clients.type AS owner_type, clients.email AS owner_email, db_users.id, data_bases.name, data_bases.type, DatabaseServers.host, DatabaseServers.port, db_users.login FROM domains, clients, db_users, data_bases, DatabaseServers WHERE db_users.db_id = data_bases.id AND data_bases.dom_id = domains.id AND clients.id = domains.cl_id AND DatabaseServers.id = data_bases.db_server_id";
		}
		$dbusers = $db->fetchAll($sql);
		$db->close();
		
		if (Util::isLinux()) {
			$cmd = $this->plesk_dir . '/bin/database';
		} else {
			$cmd = '"' . $this->plesk_dir . 'bin\database.exe"';
		}
		
		foreach ($dbusers as $dbuser) {
			$newPassword = $this->getNewPassword();
			Util::exec($cmd . ' -u ' . $dbuser['name'] . ' -update_user ' . $dbuser['login'] . ' -passwd ' . $this->wrapNewPassword($newPassword), $code);
			if ($code === 0) {
				$this->logToCsv($dbuser['owner_name'], $dbuser['owner_type'], $dbuser['owner_login'], $dbuser['owner_email'], $dbuser['domain_name'], 'dbuser', $dbuser['login'], $dbuser['id'], $newPassword);
				Log::info($dbuser['type'] . ' database ' . $dbuser['name'] . ' user on ' . $dbuser['host'] . ':' . $dbuser['port'] . ' with login ' . $dbuser['login']  . ' on domain ' . $dbuser['domain_name'] . ' New password: ' . $newPassword);
			} else {
				Log::warning('Unable to execute: ' . $cmd . ' -u ' . $dbuser['name'] . ' -update_user ' . $dbuser['login'] . ' -passwd ' . $newPassword . ' Return code is: ' . $code);
			}
		}
		
	}
	
	public function getNewPassword()
	{
		$this->getPleskPasswordStrength();
		
		$strength = array(
        		"Mediocre" => 0,
        		"Stronger" => 4,
		);

		$length = 12;
		$patterns = array();
		$patterns[] = "1234567890";
		$patterns[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
		$patterns[] = "abcdefghijklmnopqrstuvwxyz";

		$passwordString = "";
		foreach($patterns as $pattern) {
			$passwordString .= substr( str_shuffle($pattern), 0, (int) $length/count($patterns) + (int) $strength[$this->plesk_password_strength]/$strength["Stronger"]);
		}
		
		$passwordString .= substr( str_shuffle('@#%^*'), 0, $strength[$this->plesk_password_strength]);

		return str_shuffle($passwordString);
	}

	public function wrapNewPassword($newPassword)
	{
		$quotes = Util::isWindows() ? '"' : '\'';
		return $quotes . $newPassword . $quotes;
	}

	public function logToCsv($owner_name, $owner_type, $owner_login, $owner_email, $domain, $entity_type, $entity_login, $entity_id, $new_password)
	{
		$log = $owner_name . ';' . $owner_type . ';' . $owner_login . ';' . $owner_email . ';' . $domain . ';' . $entity_type . ';' . $entity_login . ';' . $entity_id . ';' . $new_password . PHP_EOL; 
		Log::write(APP_PATH . '/new_plesk_passwords.csv', $log);
	}
	
}

class PleskInstallation
{
    public function validate()
    {
        if (!self::isInstalled()) {
            Log::step('Plesk installation is not found. You will have no problems with upgrade, go on and install '.PleskVersion::getLatestPleskVersionAsString().' (http://www.parallels.com/products/plesk/)');
            return;
        }
        $this->_detectVersion();
    }
    
    static function isInstalled()
    {
        $rootPath = Util::getPleskRootPath();
        if (empty($rootPath) || !file_exists($rootPath)) {
            //Log::fatal('Plesk is not installed. Please install Plesk Panel at first.');
            return false;
        }
        return true;
    }
    
    private function _detectVersion()
    {
        Log::step('Installed Plesk version/build: ' . PleskVersion::getVersionAndBuild());
        
        $currentVersion = PleskVersion::getVersion();
        if (version_compare($currentVersion, PLESK_VERSION, 'eq')) {
            $err = 'You have already installed the latest version ' . PleskVersion::getLatestPleskVersionAsString() . '. ';
            $err .= 'Tool must be launched prior to upgrade to ' . PleskVersion::getLatestPleskVersionAsString() . ' for the purpose of getting a report on potential problems with the upgrade.';
            // TODO either introduce an option to suppress fatal error here, or always exit with 0 here.
            //Log::fatal($err);
            Log::info($err);
            exit(0);
        }
        
        if (!PleskVersion::is8x() && !PleskVersion::is9x() && !PleskVersion::is10x() && !PleskVersion::is11x() && !PleskVersion::is12x()) {
            $err = 'Unable to find Plesk 8.x, Plesk 9.x, Plesk 10.x or Plesk 12.x. ';
            $err .= 'Tool must be launched prior to upgrade to ' . PleskVersion::getLatestPleskVersionAsString() . ' for the purpose of getting a report on potential problems with the upgrade.';
            Log::fatal($err);
        }
    }
}

class PleskVersion
{
    static function is8x()
    {
        $version = PleskVersion::getVersion();
        return version_compare($version, '8.0.0', '>=') && version_compare($version, '9.0.0', '<');
    }

    static function is9x()
    {
        $version = PleskVersion::getVersion();
        return version_compare($version, '9.0.0', '>=') && version_compare($version, '10.0.0', '<');
    }

    static function is10x()
    {
        $version = PleskVersion::getVersion();
        return version_compare($version, '10.0.0', '>=') && version_compare($version, '11.0.0', '<');
    }
    
    static function is11x()
    {
    	$version = PleskVersion::getVersion();
    	return version_compare($version, '11.0.0', '>=') && version_compare($version, '12.0.0', '<');
    }

    static function is12x()
    {
    	$version = PleskVersion::getVersion();
    	return version_compare($version, '12.0.0', '>=') && version_compare($version, '18.0.500', '<');
    }
	
    static function is11_0()
    {
    	$version = PleskVersion::getVersion();
    	return version_compare($version, '11.0.0', '>=') && version_compare($version, '11.1.0', '<');
    }
    
    static function is10x_or_above()
    {
    	$version = PleskVersion::getVersion();
    	return version_compare($version, '10.0.0', '>=');
    }
    
    static function is10_1_or_below()
    {
    	$version = PleskVersion::getVersion();
    	return version_compare($version, '10.1.1', '<=');
    }
    
    static function is10_2_or_above()
    {
    	$version = PleskVersion::getVersion();
    	return version_compare($version, '10.2.0', '>=');
    }
    
    static function is10_4()
    {
    	$version = PleskVersion::getVersion();
    	return version_compare($version, '10.4.0', '>=') && version_compare($version, '10.5.0', '<');
    }
    
    static function is10_4_or_above()
    {
    	$version = PleskVersion::getVersion();
    	return version_compare($version, '10.4.0', '>=');
    }

	static function is11x_or_above()
    {
    	$version = PleskVersion::getVersion();
    	return version_compare($version, '11.0.0', '>=');
    }
	
    static function is11_5_or_above()
    {
    	$version = PleskVersion::getVersion();
    	return version_compare($version, '11.5.0', '>=');
    }
    
    static function getVersion()
    {
        $version = PleskVersion::getVersionAndBuild();
        if (!preg_match('/([0-9]+\.[0-9]+\.[0-9])/', $version, $macthes)) {
            Log::fatal("Incorrect Plesk version format. Current version: {$version}");
        }
        return $macthes[1];
    }
    
    static function getVersionAndBuild()
    {
        $versionPath = Util::getPleskRootPath().'/version';
        if (!file_exists($versionPath)) {
            Log::fatal("Plesk version file is not exists $versionPath");
        }
        $version = file_get_contents($versionPath);
        $version = trim($version);
        return $version;
    }
    
    static function getLatestPleskVersionAsString()
    {
        return 'Parallels Panel ' . PLESK_VERSION;
    }
}

class Log
{
	public function __construct()
    {
        $this->_logFile = APP_PATH . '/mass_password_reset_tool.log';
        @unlink($this->_logFile);
    }
    
    static function _getInstance()
    {
        static $_instance = null;
        if (is_null($_instance)) {
            $_instance = new Log();
        }
        return $_instance;
    }
    
    static function fatal($msg)
    {
        $log = Log::_getInstance();
        $log->_log($msg, 'FATAL_ERROR');
    }
    
    static function error($msg)
    {
        $log = Log::_getInstance();
        $log->_log($msg, 'ERROR');
    }
    
    static function warning($msg)
    {
        $log = Log::_getInstance();
        $log->_log($msg, 'WARNING');
    }
    
    static function step($msg, $useNumber=false)
    {
        static $step = 1;
        
        if ($useNumber) {
            $msg = "==> STEP " . $step . ": {$msg}";
            $step++;
        } else {
            $msg = "==> {$msg}";
        }
        
        $log = Log::_getInstance();
        $log->_log($msg, 'INFO', PHP_EOL);
    }
    
    static function resultOk()
    {
        $msg = 'Result: OK';
        Log::info($msg);
    }
    
    static function resultWarning()
    {
        $msg = 'Result: Warning';
        Log::info($msg);
    }
    
    static function resultError()
    {
        $msg = 'Result: Error';
        Log::info($msg);
    }
    
    static function info($msg)
    {
        $log = Log::_getInstance();
        $log->_log($msg, 'INFO');
    }
    
    static function dumpStatistics()
    {
        global $errors, $warnings;
        
        $str = 'Found errors: ' . $errors 
            . '; Found Warnings: ' . $warnings
        ;
        echo PHP_EOL . $str . PHP_EOL . PHP_EOL;
    }
    
    public function _log($msg, $type, $newLine='')
    {
        global $errors, $warnings;

		// TODO modern PHP (from 5.3) issues warning:
		//  PHP Warning:  date(): It is not safe to rely on the system's timezone settings. You are *required* to use the date.timezone setting 
		//  or the date_default_timezone_set() function. In case you used any of those methods and you are still getting this warning, you most 
		//  likely misspelled the timezone identifier. We selected 'America/New_York' for 'EDT/-4.0/DST' instead in 
		//  panel_preupgrade_checker.php on line 1282
    	if (getenv('AUTOINSTALLER_VERSION')) {
            $log = $newLine . "{$type}: {$msg}" . PHP_EOL;
    	} else {
			if (version_compare(PHP_VERSION, '5.3.0') >= 0) {
				date_default_timezone_set(@date_default_timezone_get());
			}
            $date = date('Y-m-d h:i:s');
            $log = $newLine . "[{$date}][{$type}] {$msg}" . PHP_EOL;
        }

        if ($type == 'ERROR' || $type == 'FATAL_ERROR') {
            $errors++;
            fwrite(STDERR, $log);
        } elseif ($type == 'WARNING') {
            $warnings++;
            fwrite(STDERR, $log);
        } elseif ($type == 'INFO') {
            //:INFO: Dump to output and write log to the file
            echo $log;
        }
        
        Log::write($this->_logFile, $log);
        
        //:INFO: Terminate the process if have the fatal error
        if ($type == 'FATAL_ERROR') {
            exit(1);
        }
    }
    
    static function write($file, $content, $mode='a+')
    {
        $fp = fopen($file, $mode);
        fwrite($fp, $content);
        fclose($fp);
    }
}

class PleskDb
{
    private $_db = null;
    
    public function __construct($dbParams)
    {
        switch($dbParams['db_type']) {
            case 'mysql':
                $this->_db = new DbMysql(
                    $dbParams['host'], $dbParams['login'], $dbParams['passwd'], $dbParams['db'], $dbParams['port']
                );
                break;
                
            case 'jet':
                $this->_db = new DbJet($dbParams['db']);
                break;
                
            case 'mssql':
                $this->_db = new DbMsSql(
                    $dbParams['host'], $dbParams['login'], $dbParams['passwd'], $dbParams['db'], $dbParams['port']
                );
                break;

            default:
                Log::fatal("{$dbParams['db_type']} is not implemented yet");
                break;
        }
    }
    
    static function getInstance()
    {
        global $options;
        static $_instance = array();
        
        $dbParams['db_type']= Util::getPleskDbType();
        $dbParams['db']     = Util::getPleskDbName();
        $dbParams['port']   = Util::getPleskDbPort();
        $dbParams['login']  = Util::getPleskDbLogin();
        $dbParams['passwd'] = $options->getDbPasswd();
        $dbParams['host']   = Util::getPleskDbHost();
        
        $dbId = md5(implode("\n", $dbParams) . microtime());
		
		$_instance[$dbId] = new PleskDb($dbParams);
        
		if (DEBUG) {
			Log::info('getInstance');
			var_dump($_instance);
		}
		
        return $_instance[$dbId];
    }
    
    public function fetchOne($sql)
    {
        if (DEBUG) {
            Log::info($sql);
        }
        return $this->_db->fetchOne($sql);
    }
    
    public function fetchRow($sql)
    {
        $res = $this->fetchAll($sql);
        if (is_array($res) && isset($res[0])) {
            return $res[0];
        }
        return array();
    }
    
    public function fetchAll($sql)
    {
        if (DEBUG) {
            Log::info($sql);
        }
        return $this->_db->fetchAll($sql);
    }
    
    public function close()
    {
    	$this->_db->close();
    }
    
    public function query($sql)
    {
    	if (DEBUG) {
            Log::info($sql);
        }
        return $this->_db->query($sql);
    }
}

class DbMysql
{
    private $_dbHandler = null;
    
    public function __construct($host, $user, $passwd, $database, $port)
    {
    	if (DEBUG) {
    		Log::info('DbMysql call');
    		var_dump($this->_dbHandler);
    	}
    	
        if ( extension_loaded('mysql') ) {
            $this->_dbHandler = @mysql_connect("{$host}:{$port}", $user, $passwd);
            if (!is_resource($this->_dbHandler)) {
                $mysqlError = mysql_error();
                if (stristr($mysqlError, 'access denied for user')) {
                    $errMsg = 'Given <password> is incorrect. ' . $mysqlError;
                } else {
                    $errMsg = 'Unable to connect database. The reason of problem: ' . $mysqlError . PHP_EOL;
                }
                $this->_logError($errMsg);
            }
            @mysql_select_db($database, $this->_dbHandler);
        } else if ( extension_loaded('mysqli') ) {
            $this->_dbHandler = @mysqli_connect($host, $user, $passwd, $database, $port);
            if (!$this->_dbHandler) {
                $mysqlError = mysqli_connect_error();
                if (stristr($mysqlError, 'access denied for user')) {
                    $errMsg = 'Given <password> is incorrect. ' . $mysqlError;
                } else {
                    $errMsg = 'Unable to connect database. The reason of problem: ' . $mysqlError . PHP_EOL;
                }
                $this->_logError($errMsg);
            }
        } else {
            Log::fatal("No MySQL extension is available");
        }
    }
    
    public function fetchAll($sql)
    {
        if ( extension_loaded('mysql') ) {
            $res = mysql_query($sql, $this->_dbHandler);
            if (!is_resource($res)) {
                $this->_logError('Unable to execute query. Error: ' . mysql_error($this->_dbHandler));
            }
            $rowset = array();
            while ($row = mysql_fetch_assoc($res)) {
                $rowset[] = $row;
            }
            return $rowset;
        } else if ( extension_loaded('mysqli') ) {
            $res = $this->_dbHandler->query($sql);
            if ($res === false) {
                $this->_logError('Unable to execute query. Error: ' . mysqli_error($this->_dbHandler));
            }
            $rowset = array();
            while ($row = mysqli_fetch_assoc($res)) {
                $rowset[] = $row;
            }
            return $rowset;
        } else {
            Log::fatal("No MySQL extension is available");
        }
        
    }
    
    public function close()
    {
    	if ( extension_loaded('mysql') ) {
    		while (is_resource($this->_dbHandler)) {
    			mysql_close($this->_dbHandler);
    		}
    	} elseif ( extension_loaded('mysqli') ) {
    		while ($link->_db->_dbHandler) {
    			$link->_db->_dbHandler->close();
    		}
   		} else {
   			Log::fatal("No MySQL extension is available");
   		}
    }
    
    public function fetchOne($sql)
    {
        if ( extension_loaded('mysql') ) {
            $res = mysql_query($sql, $this->_dbHandler);
            if (!is_resource($res)) {
                $this->_logError('Unable to execute query. Error: ' . mysql_error($this->_dbHandler));
            }
            $row = mysql_fetch_row($res);
            return $row[0];
        } else if ( extension_loaded('mysqli') ) {
            $res = $this->_dbHandler->query($sql);
            if ($res === false) {
                $this->_logError('Unable to execute query. Error: ' . mysqli_error($this->_dbHandler));
            }
            $row = mysqli_fetch_row($res);
            return $row[0];
        } else {
            Log::fatal("No MySQL extension is available");
        }
    }    
    
    public function query($sql)
    {
        if ( extension_loaded('mysql') ) {
            $res = mysql_query($sql, $this->_dbHandler);
            if ($res === false ) {
                $this->_logError('Unable to execute query. Error: ' . mysql_error($this->_dbHandler) );
            }
            return $res;
        } else if ( extension_loaded('mysqli') ) {
            $res = $this->_dbHandler->query($sql);
            if ($res === false ) {
                $this->_logError('Unable to execute query. Error: ' . mysqli_error($this->_dbHandler) );
            }
            return $res;
        } else {
            Log::fatal("No MySQL extension is available");
        }
    }    
    
    private function _logError($message)
    {
        $message = "[MYSQL ERROR] $message";
        Log::fatal($message);
    }
}

class DbClientMysql extends DbMysql
{
    private $errors = [];

    private function _logError($message)
    {
        $message = "[MYSQL ERROR] $message";
        Log::warning($message);
        $this->errors[] = $message;
    }

    function hasErrors() {
        return count($this->errors) > 0;
    }
}

class DbJet
{
    private $_dbHandler = null;
    
    public function __construct($dbPath)
    {
        $dsn = "Provider='Microsoft.Jet.OLEDB.4.0';Data Source={$dbPath}";
        $this->_dbHandler = new COM("ADODB.Connection", NULL, CP_UTF8);
        if (!$this->_dbHandler) {
            $this->_logError('Unable to init ADODB.Connection');
        }
        
        $this->_dbHandler->open($dsn);
    }
    
    public function fetchAll($sql)
    {
        $result_id = $this->_dbHandler->execute($sql);
        if (!$result_id) {
            $this->_logError('Unable to execute sql query ' . $sql);
        }
		if ($result_id->BOF && !$result_id->EOF) {
            $result_id->MoveFirst();
		}
		if ($result_id->EOF) {
		    return array();
		}
		
		$rowset = array();
		while(!$result_id->EOF) {
    		$row = array();
    		for ($i=0;$i<$result_id->Fields->count;$i++) {
                $field = $result_id->Fields($i);
                $row[$field->Name] = (string)$field->value;
    		}
    		$result_id->MoveNext();
    		$rowset[] = $row;
		}
		return $rowset;
    }
    
    public function fetchOne($sql)
    {
        $result_id = $this->_dbHandler->execute($sql);
        if (!$result_id) {
            $this->_logError('Unable to execute sql query ' . $sql);
        }
		if ($result_id->BOF && !$result_id->EOF) {
            $result_id->MoveFirst();
		}
		if ($result_id->EOF) {
		    //Log::fatal('Unable to find row');
		    return null;
		}
        $field = $result_id->Fields(0);
        $result = $field->value;
        
        return (string)$result;
    }
    
    public function query($sql)
    {
   		$result_id = $this->_dbHandler->execute($sql);
        if (!$result_id) {
            $this->_logError('Unable to execute sql query ' . $sql);
        }
    }
    
    private function _logError($message)
    {
        $message = "[JET ERROR] $message";
        Log::fatal($message);
    }
    
    public function close()
    {
		return;
    }
}

class DbMsSql extends DbJet
{
    public function __construct($host, $user, $passwd, $database, $port)
    {
        $dsn = "Provider=SQLOLEDB.1;Initial Catalog={$database};Data Source={$host}";
        $this->_dbHandler = new COM("ADODB.Connection", NULL, CP_UTF8);
        if (!$this->_dbHandler) {
            $this->_logError('Unable to init ADODB.Connection');
        }
        $this->_dbHandler->open($dsn, $user, $passwd);
    }
    
    private function _logError($message)
    {
        $message = "[MSSQL ERROR] $message";
        Log::fatal($message);
    }
    
    public function close()
    {
    	return;
    }
}

class Util
{
    static function isWindows()
    {
        if (strtoupper(substr(PHP_OS, 0, 3)) === 'WIN') {
            return true;
        }
        return false;
    }
    
    static function isLinux()
    {
        return !Util::isWindows();
    }
    
    static function isVz()
    {
        $vz = false;
        if (Util::isLinux()) {
            if (file_exists('/proc/vz/veredir')) {
                $vz = true;
            }
        } else {
            $reg = 'REG QUERY "HKLM\SOFTWARE\SWsoft\Virtuozzo" 2>nul';
            Util::exec($reg, $code);
            if ($code==0) {
                $vz = true;
            }
        }
        return $vz;
    }
    
    static function getArch()
    {
        global $arch;
        if (!empty($arch))
            return $arch;

        $arch = 'i386';
        if (Util::isLinux()) {
            $cmd = 'uname -m';
            $x86_64 = 'x86_64';
            $output = Util::exec($cmd, $code);
            if (!empty($output) && stristr($output, $x86_64)) {
                $arch = 'x86_64';
            }
        } else {
            $cmd = 'systeminfo';
            $output = Util::exec($cmd, $code);
            if (preg_match('/System Type:[\s]+(.*)/', $output, $macthes) && stristr($macthes[1], '64')) {
                $arch = 'x86_64';
            }
        }
        return $arch;
    }
    
    static function getHostname()
    {
        if (Util::isLinux()) {
            $cmd = 'hostname -f';
        } else {
            $cmd = 'hostname';
        }
        $hostname = Util::exec($cmd, $code);
        
        if (empty($hostname)) {
        	$err = 'Command: ' . $cmd . ' returns: ' . $hostname . "\n";
        	$err .= 'Hostname is not defined and configured. Unable to get hostname. Server should have properly configured hostname and it should be resolved locally.';
            Log::fatal($err);
        }
        
        return $hostname;
    }
    
    static function isFQDN($string)
    {
    	$tld_list = array(
                'aero', 'asia', 'biz', 'cat', 'com', 'coop', 'edu', 'gov', 'info', 'int', 'jobs', 'mil', 'mobi', 'museum', 'name', 'net', 
    			'org', 'pro', 'tel', 'travel', 'xxx', 'ac', 'ad', 'ae', 'af', 'ag', 'ai', 'al', 'am', 'an', 'ao', 'aq', 'ar', 'as', 'at', 
    			'au', 'aw', 'ax', 'az', 'ba', 'bb', 'bd', 'be', 'bf', 'bg', 'bh', 'bi', 'bj', 'bm', 'bn', 'bo', 'br', 'bs', 'bt', 'bv', 
    			'bw', 'by', 'bz', 'ca', 'cc', 'cd', 'cf', 'cg', 'ch', 'ci', 'ck', 'cl', 'cm', 'cn', 'co', 'cr', 'cs', 'cu', 'cv', 'cx', 
    			'cy', 'cz', 'dd', 'de', 'dj', 'dk', 'dm', 'do', 'dz', 'ec', 'ee', 'eg', 'eh', 'er', 'es', 'et', 'eu', 'fi', 'fj', 'fk', 
    			'fm', 'fo', 'fr', 'ga', 'gb', 'gd', 'ge', 'gf', 'gg', 'gh', 'gi', 'gl', 'gm', 'gn', 'gp', 'gq', 'gr', 'gs', 'gt', 'gu', 
    			'gw', 'gy', 'hk', 'hm', 'hn', 'hr', 'ht', 'hu', 'id', 'ie', 'il', 'im', 'in', 'io', 'iq', 'ir', 'is', 'it', 'je', 'jm', 
    			'jo', 'jp', 'ke', 'kg', 'kh', 'ki', 'km', 'kn', 'kp', 'kr', 'kw', 'ky', 'kz', 'la', 'lb', 'lc', 'li', 'lk', 'lr', 'ls', 
    			'lt', 'lu', 'lv', 'ly', 'ma', 'mc', 'md', 'me', 'mg', 'mh', 'mk', 'ml', 'mm', 'mn', 'mo', 'mp', 'mq', 'mr', 'ms', 'mt', 
    			'mu', 'mv', 'mw', 'mx', 'my', 'mz', 'na', 'nc', 'ne', 'nf', 'ng', 'ni', 'nl', 'no', 'np', 'nr', 'nu', 'nz', 'om', 'pa', 
    			'pe', 'pf', 'pg', 'ph', 'pk', 'pl', 'pm', 'pn', 'pr', 'ps', 'pt', 'pw', 'py', 'qa', 're', 'ro', 'rs', 'ru', 'rw', 'sa', 
    			'sb', 'sc', 'sd', 'se', 'sg', 'sh', 'si', 'sj', 'sk', 'sl', 'sm', 'sn', 'so', 'sr', 'ss', 'st', 'su', 'sv', 'sy', 'sz', 
    			'tc', 'td', 'tf', 'tg', 'th', 'tj', 'tk', 'tl', 'tm', 'tn', 'to', 'tp', 'tr', 'tt', 'tv', 'tw', 'tz', 'ua', 'ug', 'uk', 
    			'us', 'uy', 'uz', 'va', 'vc', 've', 'vg', 'vi', 'vn', 'vu', 'wf', 'ws', 'ye', 'yt', 'za', 'zm', 'zw' );
    
    	$label = '[a-zA-Z0-9\-]{1,62}\.';
    	$tld = '[\w]+';
    	if(preg_match( '/^(' . $label. ')+(' . $tld . ')$/', $string, $match ) && in_array( $match[2], $tld_list )) {
    		return TRUE;
    	} else {
    		return FALSE;
    	}
    
    }
    
    static function resolveHostname($hostname)
    {
    	$dns_record = dns_get_record($hostname, DNS_A | DNS_AAAA);

    	if (isset($dns_record[0]['ip'])) {
    		return $dns_record[0]['ip'];
    	} 
    	if (isset($dns_record[0]["ipv6"])) {
    		return $dns_record[0]['ipv6'];
    	}
    	
    	return null;
    }

    static function getIP()
    {
        $list = Util::getIPList();
        return $list[0]; //main IP
    }
    
    static function getIPList($lo=false)
    {
        if (Util::isLinux()) {
            $ifconfig = Util::lookupCommand('ifconfig');
            $output = Util::exec("{$ifconfig} -a", $code);
            if (!preg_match_all('/inet addr:([0-9\.]+)/', $output, $matches)) {
                Log::fatal('Unable to get IP address');
            }
            $ipList = $matches[1];
            foreach ($ipList as $key => $ip) {
                if (!$lo && substr($ip, 0, 3) == '127') {
                    unset($ipList[$key]);
                    continue;
                }
                trim($ip);
            }
            $ipList = array_values($ipList);
        } else {
            $cmd = 'hostname';
            $hostname = Util::exec($cmd, $code);
            $ip = gethostbyname($hostname);
            $res = ($ip != $hostname) ? true : false;
            if (!$res) {
                Log::fatal('Unable to retrieve IP address');
            }
            $ipList = array(trim($ip));
        }
        return $ipList;
    }
    
    static function getIPv6ListOnLinux()
    {
    	$ifconfig = Util::lookupCommand('ifconfig');
    	$output = Util::exec("{$ifconfig} -a", $code);
    	if (!preg_match_all('/inet6 addr: ?([^ ][^\/]+)/', $output, $matches)) {
    		return;
    	}
    	return $matches[1];
    }
    
    static function getIPv4ListOnLinux()
    {
    	$ifconfig = Util::lookupCommand('ifconfig');
    	$output = Util::exec("{$ifconfig} -a", $code);
    	if (!preg_match_all('/inet addr: ?([^ ]+)/', $output, $matches)) {
    		Log::fatal('Unable to get IP address');
    	}
    	return $matches[1];
    }
    
    static function getIPListOnWindows()
    {
    	$cmd = 'wmic.exe path win32_NetworkAdapterConfiguration get IPaddress';
    	$output = Util::exec($cmd, $code);
    	if (!preg_match_all('/"(.*?)"/', $output, $matches)) {
    		Log::fatal('Unable to get IP address');
    	}
    	return $matches[1];
    }
    
    static function getPleskRootPath()
    {
        global $_pleskRootPath;
        if (empty($_pleskRootPath)) {
            if (Util::isLinux()) {
                if (PleskOS::isDebLike()) {
                    $_pleskRootPath = '/opt/psa';
                } else {
                    $_pleskRootPath = '/usr/local/psa';
                }
            }
            if (Util::isWindows()) {
                $_pleskRootPath = Util::regPleskQuery('PRODUCT_ROOT_D', true);
            }
        }
        return $_pleskRootPath;
    }
    
    static function getPleskDbName()
    {
        $dbName = 'psa';
        if (Util::isWindows()) {
            $dbName = Util::regPleskQuery('mySQLDBName');
        }
        return $dbName;
    }
    
    static function getPleskDbLogin()
    {
        $dbLogin = 'admin';
        if (Util::isWindows()) {
            $dbLogin = Util::regPleskQuery('PLESK_DATABASE_LOGIN');
        }
        return $dbLogin;
    }
    
    static function getPleskDbType()
    {
        $dbType = 'mysql';
        if (Util::isWindows()) {
            $dbType = strtolower(Util::regPleskQuery('PLESK_DATABASE_PROVIDER_NAME'));
        }
        return $dbType;
    }
    
    static function getPleskDbHost()
    {
    	$dbHost = 'localhost';
    	if (Util::isWindows()) {
    		if (strtolower(Util::regPleskQuery('PLESK_DATABASE_PROVIDER_NAME')) == 'mysql') {
    			$dbHost = Util::regPleskQuery('MySQL_DB_HOST');
    		}
    	}
    	return $dbHost;
    }
    
    static function getPleskDbPort()
    {
        $dbPort = '3306';
        if (Util::isWindows()) {
            $dbPort = Util::regPleskQuery('MYSQL_PORT');
        }
        return $dbPort;
    }    
    
    static function regPleskQuery($key, $returnResult=false)
    {
        $arch = Util::getArch();
        if ($arch == 'x86_64') {
            $reg = 'REG QUERY "HKLM\SOFTWARE\Wow6432Node\Plesk\Psa Config\Config" /v '.$key; 
        } else {
            $reg = 'REG QUERY "HKLM\SOFTWARE\Plesk\Psa Config\Config" /v '.$key;
        }
        $output = Util::exec($reg, $code);
        
        if ($returnResult && $code!=0) {
            return false;
        }
        
        if ($code!=0) {
            Log::info($reg);
            Log::info($output);
            Log::fatal("Unable to get '$key' from registry");
        }
        if (!preg_match("/\w+\s+REG_SZ\s+(.*)/i", trim($output), $matches)) {
            Log::fatal('Unable to macth registry value by key '.$key.'. Output: ' .  trim($output));
        }
        
        return $matches[1];
    }
    
    static function regQuery($path, $key, $returnResult=false)
    {
    	$arch = Util::getArch();
    	if ($arch == 'x86_64') {
    		$reg = 'REG QUERY "HKLM\SOFTWARE\Wow6432Node' . $path .  '" '.$key; 
    	} else {
    		$reg = 'REG QUERY "HKLM\SOFTWARE' . $path .  '" '.$key;
    	}
    	$output = Util::exec($reg, $code);
    
    	if ($returnResult && $code!=0) {
    		return false;
    	}
    
    	if ($code!=0) {
    		Log::info($reg);
    		Log::info($output);
    		Log::fatal("Unable to get '$key' from registry");
    	}
    	if (!preg_match("/\s+REG_SZ\s+(.*)/i", trim($output), $matches)) {
    		Log::fatal('Unable to match registry value by key '.$key.'. Output: ' .  trim($output));
    	}
    
    	return $matches[1];
    }    
    
    static function getAutoinstallerVersion()
    {
    	if (Util::isLinux()) {
    		$rootPath = Util::getPleskRootPath();
    		$cmd = $rootPath . '/admin/sbin/autoinstaller --version';
    		$output = Util::exec($cmd, $code);
    	} else {
    		$cmd = '"' . Util::regPleskQuery('PRODUCT_ROOT_D', true) . 'admin\bin\ai.exe" --version';
    		$output = Util::exec($cmd, $code);
    	}
    	if (!preg_match("/\d+\.\d+\.\d+/", trim($output), $matches)) {
    		Log::fatal('Unable to match autoinstaller version. Output: ' .  trim($output));
    	}
    	return $matches[0];
    }
    
    static function lookupCommand($cmd, $path = '/bin:/usr/bin:/usr/local/bin:/usr/sbin:/sbin:/usr/local/sbin', $exit = true)
    {
        $dirs = explode(':', $path);
        foreach ($dirs as $dir) {
            $util = $dir . '/' . $cmd;
            if (is_executable($util)) {
                return $util;
            }
        }
        if ($exit) {
            Log::fatal("{$cmd}: command not found");
        }
    }
    
    static function getSystemDisk()
    {
    	$cmd = 'echo %SYSTEMROOT%';
    	$output = Util::exec($cmd, $code);
    	return substr($output, 0, 3);
    }
    
    static function getSystemRoot()
    {
    	$cmd = 'echo %SYSTEMROOT%';
    	$output = Util::exec($cmd, $code);
    	return $output;
    }
    
    static function getFileVersion($file)
    {
    	$fso = new COM("Scripting.FileSystemObject");
    	$version = $fso->GetFileVersion($file);
    	$fso = null;
    	return $version;
    }
    
    static function getMySQLServerVersion()
    {
    	$credentials = Util::getDefaultClientMySQLServerCredentials();
    
    	$mysql = new DbClientMysql('localhost', $credentials['admin_login'], $credentials['admin_password'] , 'information_schema', 3306);
    	if (!$mysql->hasErrors()) {
    		$sql = 'select version()';
    		$mySQLversion = $mysql->fetchOne($sql);
    		if (!preg_match("/(\d{1,})\.(\d{1,})\.(\d{1,})/", trim($mySQLversion), $matches)) {
    			Log::fatal('Unable to match MySQL server version.');
    		}
    		return $matches[0];
    	}
    }
    
    static function getDefaultClientMySQLServerCredentials()
    {
    	$db = PleskDb::getInstance();
    	$sql = "SELECT DatabaseServers.admin_login, DatabaseServers.admin_password FROM DatabaseServers WHERE type='mysql' AND host='localhost'";
    	$clientDBServerCredentials = $db->fetchAll($sql);
    	if (Util::isLinux()) {
    		$clientDBServerCredentials[0]['admin_password'] = Util::retrieveAdminMySQLDbPassword();
    	}
     	return $clientDBServerCredentials[0];
    }

	static function retrieveAdminMySQLDbPassword()
	{
		if (Util::isLinux()) {
			return trim( Util::readfile("/etc/psa/.psa.shadow") );
        }

		return null;
	}

    static function exec($cmd, &$code)
    {
	    // In PHP < 5.3 exec implies cmd /c $cmd call, but in PHP >= 5.3 -- cmd /c "$cmd" call
	    if (Util::isWindows() && version_compare(PHP_VERSION, '5.3.0') < 0)
	    {
		    $cmd = '"' . $cmd . '"'; 
		    if (DEBUG) Log::info($cmd);
	    }

        exec($cmd, $output, $code);
        return trim(implode("\n", $output));
    }

	static function readfile($file)
	{
		if (!is_file($file) || !is_readable($file))
			return null;
		$lines = file($file);
		if ($lines === false)
			return null;
		return trim(implode("\n", $lines));
	}
	
	static function readfileToArray($file)
	{
		if (!is_file($file) || !is_readable($file))
		return null;
		$lines = file($file);
		if ($lines === false)
		return null;
		return $lines;
	}
	
	static function getSettingFromPsaConf($setting)
	{
		$file = '/etc/psa/psa.conf';
		if (!is_file($file) || !is_readable($file))
			return null;
		$lines = file($file);
		if ($lines === false)
			return null;
		foreach ($lines as $line) {
			if (preg_match("/^{$setting}\s.*/", $line, $match_setting)) {
				if (preg_match("/[\s].*/i", $match_setting[0], $match_value)) {
					$value = trim($match_value[0]);
					return $value;
				}
			}
		}
		return null;
	}
	
	function GetFreeSystemMemory()
	{
		if (Util::isLinux()) {
			$cmd = 'cat /proc/meminfo';
			$output = Util::exec($cmd, $code);
			if (preg_match("/MemFree:.+?(\d+)/", $output, $MemFree)) {
				if (preg_match("/SwapFree:.+?(\d+)/", $output, $SwapFree)) {
					return $MemFree[1] + $SwapFree[1]; // returns value in Kb
				}
			}
		} else {
			$cmd = 'wmic.exe OS get FreePhysicalMemory';
			$output = Util::exec($cmd, $code);
			if (preg_match("/\d+/", $output, $FreePhysicalMemory)) {
				$cmd = 'wmic.exe PAGEFILE get AllocatedBaseSize';
				$output = Util::exec($cmd, $code);
				if (preg_match("/\d+/", $output, $SwapAllocatedBaseSize)) {
					$cmd = 'wmic.exe PAGEFILE get CurrentUsage';
					$output = Util::exec($cmd, $code);
					if (preg_match("/\d+/", $output, $SwapCurrentUsage)) {
						return $FreePhysicalMemory[0] + ($SwapAllocatedBaseSize[0] - $SwapCurrentUsage[0]) * 1000; // returns value in Kb
					}
				}
			}
		}
	}
}

class PackageMngr
{
	static function getManager($field, $package)
	{
		$redhat = 'rpm -q --queryformat \'%{' . $field . '}\n\' ' . $package;
		$debian = 'dpkg-query --show --showformat=\'${' . $field . '}\n\' '. $package;
		$suse = 'rpm -q --queryformat \'%{' . $field . '}\n\' ' . $package;
		$manager = false;
		 
		if (PleskOS::isRedHatLike()) {
			$manager = $redhat;
		} elseif (PleskOS::isDebLike()) {
			$manager = $debian;
		} elseif (PleskOS::isSuseLike()) {
			$manager = $suse;
		} else {
			return false;
		}
		 
		return $manager;
	}

	/* DPKG doesn't supports ${Release}
	 *
	*/

	static function getRelease($package)
	{
		$release = false;

		$manager = self::getManager('Release', $package);

		if (!$manager) {
			return false;
		}

		$release = Util::exec($manager, $code);
		if (!$code === 0) {
			return false;
		}
		return $release;
	}

	static function getVersion($package)
	{
		$version = false;

		$manager = self::getManager('Version', $package);

		if (!$manager) {
			return false;
		}

		$version = Util::exec($manager, $code);
		if (!$code === 0) {
			return false;
		}
		return $version;
	}

}

class PleskOS
{
    static function isSuse103()
    {
        return PleskOS::_detectOS('suse', '10.3');
    }
    
    static function isUbuntu804()
    {
        return PleskOS::_detectOS('ubuntu', '8.04');
    }
    
    static function isDebLike()
    {
    	if (PleskOS::_detectOS('ubuntu', '.*')
    	|| PleskOS::_detectOS('debian', '.*')
    	) {
    		return true;
    	}
    	return false;
    }
    
    static function isSuseLike()
    {
    	if (PleskOS::_detectOS('suse', '.*')) {
    		return true;
    	}
    	return false;
    }
    
    static function isRedHatLike()
	{
		return (PleskOS::isRedHat() || PleskOS::isCentOS() || PleskOS::isCloudLinux());
    }
    
    
    static function isRedHat()
    {
    	if (PleskOS::_detectOS('red hat', '.*')) {
    		return true;
    	}
    	return false;
    }
    
    static function isCentOS()
    {
    	if (PleskOS::_detectOS('centos', '.*')) {
    		return true;
    	}
    	return false;
	}

	static function isCloudLinux()
	{
		return PleskOS::_detectOS('CloudLinux', '.*');
	}
	
    static function isWindows2003()
    {
    	if (!Util::isWindows()) {
    		return false;
    	}
    	
    	$version = Util::regQuery('\Microsoft\Windows NT\CurrentVersion', '/v CurrentVersion', true);
    	if (version_compare($version, '5.0', '>=') && version_compare($version, '6.0', '<')) {
    		return true;
    	}
    }
    
    static function isWindows2008()
    {
    	if (!Util::isWindows()) {
    		return false;
    	}
    	 
    	$version = Util::regQuery('\Microsoft\Windows NT\CurrentVersion', '/v CurrentVersion', true);
    	if (version_compare($version, '6.0', '>=') && version_compare($version, '6.1', '<=')) {
    		return true;
    	}
    }
    
    static function _detectOS($name, $version)
    {
		$output = PleskOS::catEtcIssue();
		// This is wrong in a general case! There might be additional text between name and version.
        if (!preg_match("/{$name}[\s]+$version/i", $output)) {
            return false;
        }
        return true;
    }
     
    static function catEtcIssue()
    {
        $cmd = 'cat /etc/issue';
        $output = Util::exec($cmd, $code);
        
        return $output;
    }
    
    public function detectSystem()
    {
        Log::step('Detect system configuration');
        
        Log::info('OS: ' . (Util::isLinux() ? PleskOS::catEtcIssue() : 'Windows'));
        Log::info('Arch: ' . Util::getArch());
    }
}

class PleskValidator
{
    static function isValidIp($value)
    {
        if (!is_string($value)) {
            return false;
        }
        if (!PleskValidator::validateIPv4($value) && !PleskValidator::validateIPv6($value)) {
            return false;
        }
        return true;
    }
    
    static function validateIPv4($value)
    {
        $ip2long = ip2long($value);
        if ($ip2long === false) {
            return false;
        }

        return $value == long2ip($ip2long);
    }

    static function validateIPv6($value)
    {
        if (strlen($value) < 3) {
            return $value == '::';
        }

        if (strpos($value, '.')) {
            $lastcolon = strrpos($value, ':');
            if (!($lastcolon && PleskValidator::validateIPv4(substr($value, $lastcolon + 1)))) {
                return false;
            }

            $value = substr($value, 0, $lastcolon) . ':0:0';
        }

        if (strpos($value, '::') === false) {
            return preg_match('/\A(?:[a-f0-9]{1,4}:){7}[a-f0-9]{1,4}\z/i', $value);
        }

        $colonCount = substr_count($value, ':');
        if ($colonCount < 8) {
            return preg_match('/\A(?::|(?:[a-f0-9]{1,4}:)+):(?:(?:[a-f0-9]{1,4}:)*[a-f0-9]{1,4})?\z/i', $value);
        }

        // special case with ending or starting double colon
        if ($colonCount == 8) {
            return preg_match('/\A(?:::)?(?:[a-f0-9]{1,4}:){6}[a-f0-9]{1,4}(?:::)?\z/i', $value);
        }
        
        return false;
    }
}

class CheckRequirements
{
    public function validate()
    {
        if (!PleskInstallation::isInstalled()) {
            //:INFO: skip chking mysql extension if plesk is not installed
            return;
        }
         
        $reqExts = array();
        foreach ($reqExts as $name) {
            $status = extension_loaded($name);
            if (!$status) {
                $this->_fail("PHP extension {$name} is not installed");
            }
        }
    }
    
    private function _fail($errMsg)
    {
        echo '===Checking requirements===' . PHP_EOL;
        echo PHP_EOL . 'Error: ' . $errMsg . PHP_EOL;
        exit(1);
    }
}

class GetOpt
{
    public $_argv = null;
	public $_adminDbPasswd = null;
	public $_newAdminPasswd = '';
    
	public function __construct()
    {
        $this->_argv = $_SERVER['argv'];
		if (empty($this->_argv[1]) && Util::isLinux()) {
			$this->_adminDbPasswd = Util::retrieveAdminMySQLDbPassword();
		} else {
			$this->_adminDbPasswd = $this->_argv[1];
		}
		if (!empty($this->_argv[2]) && !preg_match('/^--/', $this->_argv[2])) {
			$this->_newAdminPasswd = $this->_argv[2];
		}
		if (!strpos(implode(' ', $this->_argv), ' --')) {
			$this->_argv[] = '--all'; // if there is no any arguments like --something, than treat this as --all 
		}
    }
    
    public function validate()
    {
        if (empty($this->_adminDbPasswd) && PleskInstallation::isInstalled()) {
            echo 'Please specify Plesk database password';
            $this->_helpUsage();
        }
        if (in_array('-h', $this->_argv) || in_array('--help', $this->_argv)) {
        	$this->_helpUsage();
        }
    }
    
    public function getDbPasswd()
    {
        return $this->_adminDbPasswd;
    }
    
    public function getNewAdminPasswd()
    {
    	return $this->_newAdminPasswd;
    }
    
    private function _helpUsage()
    {
        echo PHP_EOL . "Usage: {$this->_argv[0]} <plesk_db_admin_password> [new_plesk_db_admin_password] [options]" . PHP_EOL;
        echo "Options: --all - [default] change passwords for all supported entities and clean up sessions table in Plesk database" . PHP_EOL;
        echo "         --admin - change password for admin" . PHP_EOL;
		echo "         --apsc - change password for apsc database" . PHP_EOL;
        echo "         --clean-up-sessions - clean up sessions table in Plesk database" . PHP_EOL;
        echo "         --additionaladmins - change passwords for additional administrators accounts" . PHP_EOL; 
        echo "         --resellers - change passwords for resellers" . PHP_EOL;
        echo "         --clients - change passwords for clients" . PHP_EOL;
        echo "         --domains - change passwords for main FTP account of domains" . PHP_EOL;
        echo "         --domainadmins - change passwords for Domain Administrators" . PHP_EOL;
        echo "         --users - change passwords for hosting panel users" . PHP_EOL;
        echo "         --additionalftpaccounts - change passwords for additional FTP accounts for domains" . PHP_EOL;
        echo "         --subdomains - change passwords for subdomains. NOTE: For Plesk 10.x subdomains treated as domains." . PHP_EOL;
        echo "         --dbusers - change passwords for database users." . PHP_EOL;
        echo "         --webusers - change passwords for webusers" . PHP_EOL;
        echo "         --mailaccounts - change passwords for mail accounts" . PHP_EOL;
		echo "         --pdusers - change passwords for protected directories users" . PHP_EOL;
        exit(1);
    }
}

$errors = $warnings = 0;

//:INFO: Validate Plesk installation
$pleskInstallation = new PleskInstallation();
$pleskInstallation->validate();

//:INFO: Detect system
$pleskOs = new PleskOS();
$pleskOs->detectSystem();

//:INFO: Need to make sure that given db password is valid
if (PleskInstallation::isInstalled()) {
    Log::step('Validate given db password');
    $pleskDb = PleskDb::getInstance();
    Log::resultOk();
}

//:INFO: Dump script version
Log::step('Plesk Password Changer version: ' . PRE_UPGRADE_SCRIPT_VERSION);

//:INFO: Validate known OS specific issues with recommendation to avoid bugs in Plesk
$PleskPasswordChanger = new PleskPasswordChanger();
$PleskPasswordChanger->changeAllPasswords($options->_newAdminPasswd);

Log::dumpStatistics();

if ($errors > 0 || $warnings > 0) {
	exit(1);
}