<?php
/**
 * @Copyright © 2002-2018 Acronis International GmbH. All rights reserved.
 */

use Modules_AcronisBackup_Context as Context;
use Modules_AcronisBackup_Helpers_Arr as Arr;
use Modules_AcronisBackup_PleskApiClient as PleskApiClient;
use Modules_AcronisBackup_PleskApiResponseException as PleskApiResponseException;
use Modules_AcronisBackup_Srv_Client as SrvClient;
use Modules_AcronisBackup_MetaDataWriter as MetaDataWriter;

class Modules_AcronisBackup_MetaDataCollector
{
    private $isMailServiceEnabled = true;
    private $customersToWrite = 10;
    protected $metadataWriter;

    public function __construct()
    {
        $this->metadataWriter = new MetaDataWriter();
    }

    public function collectMetaData()
    {
        $logPrefix = 'Collecting metadata. ';

        $this->metadataWriter->write($this->collectExtensionInfo(), 'w');
        pm_Log::info($logPrefix . 'Extension info: OK');

        $pleskVersion = $this->collectPleskInfo();
        $this->metadataWriter->write($pleskVersion);
        $reportData = $pleskVersion;
        pm_Log::info($logPrefix . 'Plesk info: OK');

        // Customers data is written by chunks to the file.
        $reportData = array_merge($reportData, $this->collectCustomersData());
        pm_Log::info($logPrefix . 'Customers data: OK');

        $dbServers = $this->collectDbServers();
        $this->metadataWriter->write($dbServers);
        pm_Log::info($logPrefix . 'DB servers: OK');

        $this->metadataWriter->write(
            [
                'timestamp' => (new DateTime())
                    ->setTimezone(new DateTimeZone('UTC'))
                    ->getTimestamp()
            ]
        );

        $this->metadataWriter->postProcessMeta();
        $this->reportAnalyticsData($reportData);
    }

    private function collectPleskInfo()
    {
        $plesk = [
            'version' => pm_ProductInfo::getVersion(),
        ];
        return compact('plesk');
    }

    private function collectCustomersData()
    {
        $pleskApiClient = new PleskApiClient();
        $customers = [];
        $counter = 0;
        $mysqlDbsNumber = 0;
        $fetchedCustomers = [];

        // Version limitation for pm_Client::getAll() function.
        if (version_compare(pm_ProductInfo::getVersion(), '17.8.4', '>=')) {
            $fetchedClients = pm_Client::getAll();
            foreach ($fetchedClients as $client) {
                $fetchedCustomers[] = [
                    'id' => $client->getId()
                ];
            }
        } else {
            $resellers = $pleskApiClient->getResellers();
            $fetchedCustomers = $pleskApiClient->fetchCustomers();
            foreach ($resellers as $reseller) {
                $fetchedCustomers[] = ['id' => $reseller['id']];
            }
            $fetchedCustomers[] = [
                'id' => 1
            ];
        }

        $customersLength = count($fetchedCustomers);
        foreach ($fetchedCustomers as $customer) {
            $webspaces = $pleskApiClient->fetchWebspaces(['owner-id' => $customer['id']]);
            foreach ($webspaces as $webspace) {
                $webspace['pmDomain'] = pm_Domain::getByDomainId($webspace['id']);
                $webspaceDomain = $this->buildDomainInfo($webspace['pmDomain']);
                $websapaceDomains = $pleskApiClient->fetchDomainsForWebspace($webspace['id']);
                $domains = [$webspaceDomain['id'] => $webspaceDomain];
                $subdomains = [];
                foreach ($websapaceDomains as $domain) {
                    $pmDomain = pm_Domain::getByDomainId($domain['id']);
                    $domainInfo = $this->buildDomainInfo($pmDomain);
                    if ((int)$domainInfo['parentDomainId'] === 0) {
                        $domains[$domain['id']] = $domainInfo;
                    } else {
                        $subdomains[$domainInfo['parentDomainId']][$domainInfo['id']] = $domainInfo;
                    }
                }
                $databases = $pleskApiClient->fetchDatabasesForWebspace($webspace['id']);
                foreach ($databases as $dbData) {
                    if ($dbData['type'] === PleskApiClient::DBMS_MYSQL) {
                        $mysqlDbsNumber += 1;
                    }
                }
                $mailboxes = $this->fetchMailboxesForDomains($domains + $subdomains);
                // merge subdomains with main domains, it's possible to have mailboxes on subdomains
                foreach ($subdomains as $parentDomainId => $subdomainsList) {
                    $domains[$parentDomainId]['subdomains'] = $subdomainsList;
                }

                $webspace = array_merge($webspace, [
                    'domains' => $domains,
                    'databases' => Arr::combineByKey($databases, 'id'),
                    'mailboxes' => Arr::combineByKey($mailboxes, 'id'),
                ]);

                $pmClient = $webspace['pmDomain']->getClient();
                unset($webspace['pmDomain']);
                if (!array_key_exists($pmClient->getId(), $customers)) {
                    $customers[$pmClient->getId()] = [
                        'id' => $pmClient->getId(),
                        'guid' => $pmClient->getProperty('guid'),
                        'login' => $pmClient->getProperty('login'),
                        'email' => $pmClient->getProperty('email'),
                        'pname' => $pmClient->getProperty('pname'),
                        'cr_date' => $pmClient->getProperty('cr_date'),
                        'is_customer' => !$pmClient->isAdmin() && !$pmClient->isReseller(),
                        'webspaces' => [],
                    ];
                }
                $customers[$pmClient->getId()]['webspaces'][$webspace['id']] = $webspace;
            }

            $counter++;
            if ($counter % $this->customersToWrite === 0 || $customersLength === $counter) {
                $customers = Arr::combineByKey($customers, 'id');
                if (!empty($customers)) {
                    $this->metadataWriter->write(compact('customers'));
                }
                $customers = [];
            }
        }
        return compact('customersLength', 'mysqlDbsNumber');
    }

    private function buildDomainInfo(pm_Domain $pmDomain)
    {
        $id = $pmDomain->getId();
        $name = $pmDomain->getName();
        try {
            $documentRoot = $pmDomain->getDocumentRoot();
        } catch (Exception $e) {
            $documentRoot = null;
        }
        try {
            $parentDomainId = (int)$pmDomain->getProperty('parentDomainId');
        } catch (Exception $e) {
            $parentDomainId = 0;
        }
        try {
            $systemUserLogin = $pmDomain->getSysUserLogin();
        } catch (PleskFatalException $e) {
            $systemUserLogin = null;
            $parentDomainId = 0;
        }
        try {
            $homePath = $pmDomain->getHomePath();
        } catch (PleskFatalException $e) {
            pm_Log::warn("Site \"{$name}\" ({$id}) does not have physical hosting");
            $homePath = null;
        }
        try {
            $vhostSystemPath = $pmDomain->getVhostSystemPath();
        } catch(Exception $e) {
            $vhostSystemPath  = null;
        }
        if (!is_null($systemUserLogin)) {
            $systemUserUID = posix_getpwnam($systemUserLogin)['uid'];
            $systemUserGID = posix_getpwnam($systemUserLogin)['gid'];
        } else {
            $systemUserUID = null;
            $systemUserGID = null;
        }

        return compact(
            'id',
            'name',
            'homePath',
            'vhostSystemPath',
            'systemUserLogin',
            'systemUserUID',
            'systemUserGID',
            'documentRoot',
            'parentDomainId'
        );
    }

    private function collectExtensionInfo()
    {
        $extension = [
            'id' => pm_Context::getModuleId(),
            'version' => Context::getExtensionVersion(),
            'release' => Context::getExtensionReleaseNumber(),
        ];
        return compact('extension');
    }

    private function collectDbServers()
    {
        $pleskApiClient = new PleskApiClient();
        $dbServers = $pleskApiClient->fetchDbServers();

        $dbServers = array_map(function (array $dbServer) {
            $dbmsOpts = [];

            if ($dbServer['isLocal'] && $dbServer['type'] === PleskApiClient::DBMS_MYSQL) {
                $dbmsOpts['passwordPath'] = Context::getMySqlAdminPasswordPath();

                try {
                    $dbmsConnectionOpts = [
                        'host' => $dbServer['host'],
                        'port' => $dbServer['port'],
                        'username' => $dbServer['admin'],
                        'password' => trim(@file_get_contents($dbmsOpts['passwordPath'])),
                        'dbname' => '',
                    ];

                    $dbAdapter = Zend_Db::factory('Pdo_Mysql', $dbmsConnectionOpts);

                    $mysqlVars = $dbAdapter->fetchPairs('SHOW VARIABLES WHERE Variable_Name = "datadir";');
                    if (isset($mysqlVars['datadir'])) {
                        $dbmsOpts['datadir'] = $mysqlVars['datadir'];
                    }

                    $mysqlVersion = $dbAdapter->fetchOne("SELECT VERSION()");
                    if ($mysqlVersion) {
                        $dbmsOpts['version'] = $mysqlVersion;
                    }
                } catch (Exception $e) {
                    pm_Log::err('Unable to obtain data from mysql');
                    pm_Log::err($e);
                }
            }

            $dbServer['dbmsOptions'] = $dbmsOpts;
            return $dbServer;
        }, $dbServers);

        $dbServers = Arr::combineByKey($dbServers, 'id');
        return compact('dbServers');
    }

    private function fetchMailboxesForDomains(array $domains)
    {
        if (!$this->isMailServiceEnabled) {
            return [];
        }

        $mailboxes = [];
        $pleskApiClient = new PleskApiClient();
        try {
            foreach ($domains as $domain) {
                $mailboxes = array_merge(
                    $mailboxes,
                    $pleskApiClient->fetchMailboxesForDomain($domain['id'])
                );
            }
        } catch (PleskApiResponseException $e) {
            if ($e->isComponentError()) {
                pm_Log::warn('Unable to fetch mailboxes, component is not installed or not configured.');
                $this->isMailServiceEnabled = false;
                return [];
            }
            throw $e;
        }
        return $mailboxes;
    }

    private function reportAnalyticsData($reportData)
    {
        (new SrvClient())->reportAnalyticsEvent(
            SrvClient::ANALYTICS_EVENT_CATEGORY_PLESK_CUSTOMERS,
            SrvClient::ANALYTICS_EVENT_ACTION_COLLECT,
            null,
            $reportData['customersLength']
        );

        (new SrvClient())->reportAnalyticsEvent(
            SrvClient::ANALYTICS_EVENT_CATEGORY_MYSQL_DATABASES,
            SrvClient::ANALYTICS_EVENT_ACTION_COLLECT,
            null,
            $reportData['mysqlDbsNumber']
        );
        (new SrvClient())->reportAnalyticsEvent(
            SrvClient::ANALYTICS_EVENT_CATEGORY_PLESK_VERSION,
            SrvClient::ANALYTICS_EVENT_ACTION_COLLECT,
            $reportData['plesk']['version']
        );
    }
}
